<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Admin;

use R;
use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagTable;
use Erlage\Photogram\Data\Tables\Post\PostCommentTable;
use Erlage\Photogram\Data\Dtos\Admin\AdminTableStatsDTO;

final class AdminContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function userSearch(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::$flagAdminMode = true;

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $searchTextFromReq = self::$request -> findKey(
                UserTable::DISPLAY_NAME,
                RequestTable::PAYLOAD,
                UserTable::TABLE_NAME
            );

            $offset = self::$request -> findKeyOffset(UserTable::ID, UserTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $searchTextFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::adminEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $userTableQuery = (new Query()) -> from(UserTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | select search text matches
            |--------------------------------------------------------------------------
            */

            $userTableQuery
                -> openParenthesis()
                -> whereContains(UserTable::USERNAME, $searchTextFromReq)
                -> or()
                -> whereContains(UserTable::DISPLAY_NAME, $searchTextFromReq)
                -> closeParenthesis();

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $userTableQuery -> greaterThan(UserTable::ID, $offset);
            }
            else
            {
                $userTableQuery -> lessThan(UserTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $userTableQuery
                -> orderByDesc(UserTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_USER_SEARCH));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $userBeans = $userTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($userBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | prepare maps
            |--------------------------------------------------------------------------
            */

            self::processBeans(UserTable::getTableName(), $userBeans);
        });
    }

    public static function loadUser(): void
    {
        self::$flagAdminMode = true;

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | ensure admin is authenticated
            |--------------------------------------------------------------------------
            */

            self::adminEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userIdFromReq = self::$request -> findKey(
                UserTable::ID,
                RequestTable::PAYLOAD,
                UserTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure target model exists
            |--------------------------------------------------------------------------
            */

            $targetUserModel = UserModel::findFromId_noException($userIdFromReq);

            self::ensureModel($targetUserModel, ResponseConstants::ENTITY_NOT_FOUND_MSG);

            /*
            |--------------------------------------------------------------------------
            | add to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), $targetUserModel -> getDataMap());
        });
    }

    public static function dashboardMarkUp(): void
    {
        self::$flagAdminMode = true;

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | ensure admin is authenticated
            |--------------------------------------------------------------------------
            */

            self::adminEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | get stats
            |--------------------------------------------------------------------------
            */

            $tables = array(
                UserTable::erlClass(),
                PostTable::erlClass(),
                PostLikeTable::erlClass(),
                PostCommentTable::erlClass(),
                HashtagTable::erlClass(),
            );

            $monthDates = array();
            $monthLabels = array();
            $subQueries = array();

            // prepare dates and months

            for ($i = 5; $i >= -1; $i--)
            { // this is for seven months
                \array_push($monthLabels, \date('M', \strtotime("-{$i} month", \time())));

                switch (DATABASE_TYPE)
                {
                    case DATABASE_TYPE_MYSQL:
                    case DATABASE_TYPE_MARIA_DB:
                        \array_push($monthDates, \date('y-m-01 00:00:00', \strtotime("-{$i} month", \time())));

                    break;

                    case DATABASE_TYPE_POSTGRESQL:
                        \array_push($monthDates, \date('m-01-y 00:00:00', \strtotime("-{$i} month", \time())));

                    break;
                }
            }

            // discard seventh month, but use its date in collecting data for sixth month(current)
            \array_pop($monthLabels);

            // prepare raw query

            foreach ($monthLabels as $key => $month)
            {
                foreach ($tables as $tableClass)
                {
                    $tableName = $tableClass::getTableName();
                    $tablePrimaryAttribute = $tableClass::getPrimaryAttribute();
                    $tableRegistrationStampAttribute = $tableClass::getRegistrationStampAttribute();

                    switch (DATABASE_TYPE)
                    {
                        case DATABASE_TYPE_MYSQL:
                        case DATABASE_TYPE_MARIA_DB:
                            $subQueries[] = "( SELECT COUNT({$tablePrimaryAttribute}) FROM {$tableName}
                                WHERE
                                    {$tableRegistrationStampAttribute} > '{$monthDates[$key]}'
                                    AND {$tableRegistrationStampAttribute} < '{$monthDates[$key + 1]}'
                            ) AS {$tableName}_{$month} ";

                        break;

                        case DATABASE_TYPE_POSTGRESQL:
                            $subQueries[] = "( SELECT COUNT({$tablePrimaryAttribute}) FROM \"{$tableName}\"
                                WHERE
                                \"{$tableRegistrationStampAttribute}\" > '{$monthDates[$key]}'
                                    AND \"{$tableRegistrationStampAttribute}\" < '{$monthDates[$key + 1]}'
                            ) AS \"{$tableName}_{$month}\" ";

                        break;
                    }
                }
            }

            // for all time stats

            foreach ($tables as $tableClass)
            {
                $tableName = $tableClass::getTableName();
                $tablePrimaryAttribute = $tableClass::getPrimaryAttribute();

                switch (DATABASE_TYPE)
                {
                    case DATABASE_TYPE_MYSQL:
                    case DATABASE_TYPE_MARIA_DB:
                        $subQueries[] = "( SELECT COUNT({$tablePrimaryAttribute}) FROM {$tableName}) AS {$tableName}_total ";

                    break;

                    case DATABASE_TYPE_POSTGRESQL:
                        $subQueries[] = "( SELECT COUNT(\"{$tablePrimaryAttribute}\") FROM \"{$tableName}\") AS \"{$tableName}_total\" ";

                    break;
                }
            }

            // get data

            $rawQuery = 'SELECT ' . \implode(',', $subQueries);

            $dataRows = R::getAll($rawQuery);

            // there's only one row

            $stats = \array_pop($dataRows);

            /*
            |--------------------------------------------------------------------------
            | do some computations
            |--------------------------------------------------------------------------
            */

            $lastMonthPercentages = array();

            foreach ($tables as $tableClass)
            {
                $tableName = $tableClass::getTableName();

                $total = $stats["{$tableName}_total"];
                $fromLastMonth = $stats["{$tableName}_{$monthLabels[4]}"];

                $lastMonthPercentages[$tableName] = $total > 0 ? \round($fromLastMonth / $total * 100, 2) : 0;
            }

            /*
            |--------------------------------------------------------------------------
            | prepare response
            |--------------------------------------------------------------------------
            */

            /**
             * @var AdminTableStatsDTO[]
             */
            $responseDTOs = array();

            foreach ($tables as $tableClass)
            {
                $tableName = $tableClass::getTableName();

                $monthlyStats = array();

                foreach ($monthLabels as $key => $label)
                {
                    $monthlyStats[$label] = $stats["{$tableName}_{$label}"];
                }

                $responseDTOs[] = (new AdminTableStatsDTO())
                    -> setTableName($tableName)
                    -> setTotalRows((string) \number_format((int) ($stats["{$tableName}_total"])))
                    -> setPreviousSixMonthStats($monthlyStats)
                    -> setPercentagFromLastMonth((string) $lastMonthPercentages[$tableName]);
            }

            self::$response -> setContent(AdminTableStatsDTO::DTO_NAME, $responseDTOs);
        });
    }
}
